<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
if (!osf_is_woocommerce_activated()) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Repeater;

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class OSF_Elementor_Products_Categories_Carousel extends OSF_Elementor_Carousel_Base {

    public function get_categories() {
        return array('opal-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'opal-product-categories-carousel';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return __('Opal Product Categories Carousel', 'huntor-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-tabs';
    }


    /**
     * Register tabs widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {

        //Section Query
        $this->start_controls_section(
            'section_setting',
            [
                'label' => __('Settings', 'huntor-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title_name',
            [
                'label'   => __('Alternate Name', 'huntor-core'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'Alternate Name',
            ]
        );

        $repeater->add_control(
            'image',
            [
                'label'      => __('Choose Image', 'huntor-core'),
                'default'    => [
                    'url' => Elementor\Utils::get_placeholder_image_src(),
                ],
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
            ]
        );

        $repeater->add_control(
            'categories',
            [
                'label'    => __('Categories', 'huntor-core'),
                'type'     => Controls_Manager::SELECT2,
                'options'  => $this->get_product_categories(),
                'multiple' => false,
            ]
        );

        $this->add_control(
            'product_cat_items',
            [
                'label'       => __('Item', 'huntor-core'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '{{{ title_name }}}'
            ]
        );

        $this->add_responsive_control(
            'column',
            [
                'label'   => __('Columns', 'huntor-core'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'default' => 3,
                'options' => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 6 => 6],
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `testimonial_image_size` and `testimonial_image_custom_dimension`.
                'default'   => 'full',
                'separator' => 'none',
            ]
        );

        $this->add_responsive_control(
            'gutter',
            [
                'label'     => __('Gutter', 'huntor-core'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cats-carousel-wrapper'              => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
                    '{{WRAPPER}} .product-cats-carousel-wrapper .column-item' => 'padding-left: calc({{SIZE}}{{UNIT}} / 2); padding-right: calc({{SIZE}}{{UNIT}} / 2); padding-bottom: calc({{SIZE}}{{UNIT}})',
                ],
            ]
        );

        $this->end_controls_section();

        //STYLE
        $this->add_control_carousel();

        //STYLE
        $this->start_controls_section(
            'section_wrapper_style',
            [
                'label' => __('Content', 'huntor-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'wrapper_pr_cate_padding',
            [
                'label'      => __('Padding', 'huntor-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cats-meta' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'      => 'category_name_typography',
                'selector'  => '{{WRAPPER}} .cats-title',
                'label'     => 'Typography',
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'name_spacing',
            [
                'label'      => __('Spacing', 'huntor-core'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'size_units' => ['px'],
                'selectors'  => [
                    '{{WRAPPER}} .cats-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_name_style');

        $this->start_controls_tab(
            'tab_name_normal',
            [
                'label' => __('Normal', 'huntor-core'),
            ]
        );

        $this->add_control(
            'name_color',
            [
                'label'     => __('Color', 'huntor-core'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product-cats-carousel-item-wrapper .cats-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_name_hover',
            [
                'label' => __('Hover', 'huntor-core'),
            ]
        );

        $this->add_control(
            'name_color_hover',
            [
                'label'     => __('Color', 'huntor-core'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product-cats-carousel-item-wrapper .cats-title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tabs();

        $this->add_control(
            'total_heading',
            [
                'label'     => __('Categories', 'huntor-core'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'total_color',
            [
                'label'     => __('Categories Color', 'huntor-core'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cats-total' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'total_name_typography',
                'selector' => '{{WRAPPER}} .cats-total',
                'label'    => 'Typography'
            ]
        );


        $this->end_controls_section();

    }


    protected function get_product_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }
        return $results;
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute('item', 'class', 'product-cats-carousel-item');
        $this->add_render_attribute('item', 'class', 'column-item');

        // Row
        $this->add_render_attribute('row', 'class', 'row');
        if ($settings['enable_carousel'] === 'yes') {
            $this->add_render_attribute('row', 'class', 'owl-carousel owl-theme');
            $carousel_settings = $this->get_carousel_settings();
            $this->add_render_attribute('row', 'data-settings', wp_json_encode($carousel_settings));
        } else {
            $this->add_render_attribute('row', 'data-elementor-columns', $settings['column']);
            if (!empty($settings['column_tablet'])) {
                $this->add_render_attribute('row', 'data-elementor-columns-tablet', $settings['column_tablet']);
            }
            if (!empty($settings['column_mobile'])) {
                $this->add_render_attribute('row', 'data-elementor-columns-mobile', $settings['column_mobile']);
            }
        }

        $category = $settings['product_cat_items'][0]['categories'];

        ?>

        <div class="product-cats-carousel-wrapper">
            <div <?php echo $this->get_render_attribute_string('row') ?>>

                <?php foreach ($settings['product_cat_items'] as $item):
                    if ($item['categories']) {
                        $category = get_term_by('slug', $item['categories'], 'product_cat');
                    }
                    if (!is_wp_error($category)) {
                        if (!empty($item['image']['id'])) {
                            $item['image_size']             = $settings['image_size'];
                            $item['image_custom_dimension'] = $settings['image_custom_dimension'];
                            $image                          = Group_Control_Image_Size::get_attachment_image_html($item, 'image');
                        } else {
                            $thumbnail_id = get_term_meta($category->term_id, 'thumbnail_id', true);
                            if (!empty($thumbnail_id)) {
                                $image = '<img src="' . wp_get_attachment_url($thumbnail_id) . '" alt="" >';
                            } else {
                                $image = '<img src="' . wc_placeholder_img_src() . '" alt="" >';
                            }
                        }

                        ?>
                        <div <?php echo $this->get_render_attribute_string('item'); ?>>
                            <div class="product-cats-carousel-item-wrapper">
                                <div class="cats-image">
                                    <?php echo $image; ?>
                                    <a href="<?php echo esc_url(get_term_link($category)); ?>"></a>
                                </div>
                                <div class="product-cats-meta">
                                    <a class="link_category_product"
                                       href="<?php echo esc_url(get_term_link($category)); ?>"
                                       title="<?php echo esc_attr($category->name); ?>">
                                        <span class="screen-reader-text"><?php echo esc_html($category->name); ?></span>
                                    </a>

                                    <div class="cats-title">
                                        <a href="<?php echo esc_url(get_term_link($category)); ?>"
                                           title="<?php echo esc_attr($category->name); ?>">
                                            <span class="cats-title-text"><?php echo empty($item['title_name']) ? esc_html($category->name) : wp_kses_post($item['title_name']); ?></span>
                                        </a>
                                    </div>

                                    <div class="cats-total">
                                        <?php echo esc_html($category->count) . esc_html__(' items', 'huntor-core'); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php
                    }
                endforeach;
                ?>
            </div>
        </div>
        <?php
    }

}

$widgets_manager->register(new OSF_Elementor_Products_Categories_Carousel());

